--========================================================--
-- Unseasonal Weather - Radio Forecast Scheduler (Server)
-- Daily roll, schedule hour/minute, enqueue forecast in ModData.
--========================================================--

if not UW_RF then UW_RF = {} end

require("UW_RF_State")
require("UW_RF_Messages")

local function cfg()
    local sv = SandboxVars and SandboxVars.UW_RadioForecasting or {}
    return {
        enabled          = sv.Enabled ~= false,
        -- Probabilidad diaria fija de emitir (sin configuración en sandbox): 80% => ~20% de dias en silencio
        forecastChance   = 80,
        falseChance      = tonumber(sv.FalseForecastChance) or 10,
        useNightWindow   = sv.UseNightWindow == true,
        debug            = sv.DebugRadioForecast == true or sv.DebugLogging == true,
    }
end

local function pick(poolName)
    local pool = (UW_RF.MessageKeys and UW_RF.MessageKeys[poolName]) or nil
    if not pool or #pool == 0 then return nil end
    return pool[ZombRand(#pool) + 1]
end

local function pickType()
    local roll = ZombRand(100) + 1 -- 1..100

    if roll <= 3 then
        return "INTERRUPTED"        -- 3%
    elseif roll <= 23 then
        return "FALSE"              -- 20%
    elseif roll <= 43 then
        return "ERRONEOUS"          -- 20%
    else
        return "NORMAL"             -- 57%
    end
end

local function pickTime(useNight)
    if useNight then
        return ZombRand(20, 24), ZombRand(0, 60) -- 20-23h
    end
    return ZombRand(8, 12), ZombRand(0, 60) -- 8-11h
end

local function scheduleForDay(dayIndex, worldHours)
    local c = cfg()
    if not c.enabled then return end

    local roll = ZombRand(100)
    if roll >= c.forecastChance then
        if c.debug then
            print(string.format("[UW_RF][SCHED] Silence today (roll=%d threshold=%d)", roll, c.forecastChance))
        end
        local mdSilence = UW_RF_GetState()
        mdSilence.lastScheduledDay = dayIndex
        UW_RF_SaveState()
        return
    end

    local msgType = pickType()
    local key = pick(msgType)
    if not key then
        if c.debug then print("[UW_RF][SCHED] No key found for type " .. tostring(msgType)) end
        return
    end

    local hour, minute = pickTime(c.useNightWindow)
    local scheduledWorldHours = (dayIndex * 24) + hour + ((minute or 0) / 60)
    local expiresAt = scheduledWorldHours + 2.0 -- vence 2h despues de la hora programada

    local pending = {
        day      = dayIndex,
        hour     = hour,
        minute   = minute,
        key      = key,
        cat      = msgType,
        walkie    = true,
        walkieKey = pick("NORMAL"), -- reuse as hint
        expires   = expiresAt,
    }
    -- Debug-only text (no emision desde aqui)
    pending.text = getText(key)

    local md = UW_RF_GetState()
    md.pending = pending
    md.lastScheduledDay = dayIndex
    UW_RF_SaveState()

    if c.debug then
        print(string.format("[UW_RF][SCHED] Scheduled %s at %02d:%02d (day %d) key=%s expires=%.2f", msgType, hour, minute, dayIndex, key, expiresAt))
    end
end

local function tick()
    local c = cfg()
    if not c.enabled then return end

    local gt = getGameTime()
    if not gt then return end

    local worldHours = gt:getWorldAgeHours()
    local dayIndex   = math.floor(worldHours / 24)

    local md = UW_RF_GetState()
    if md.pending and md.pending.day == dayIndex then
        return
    end

    if md.lastScheduledDay ~= dayIndex then
        scheduleForDay(dayIndex, worldHours)
    end
end

Events.OnGameStart.Add(tick)
Events.OnInitWorld.Add(tick)
Events.EveryTenMinutes.Add(tick)

print("[UW_RF] Radio Forecast Scheduler (server) loaded.")
